! $Header: $
! $Name:   $
      MODULE MITGCM_ORG_ATM_ESMF_DRIVER

      USE ESMF_MOD
      USE MITGCM_ORG_ESMF_UTILS
      USE MITGCM_ORG_ATM_ESMF_EXPORTS
      USE MITGCM_ORG_ATM_ESMF_IMPORTS

      USE MITGCM_ORG_ATM, ONLY: ATM_INIT => DRIVER_INIT
      USE MITGCM_ORG_ATM, ONLY: ATM_RUN  => DRIVER_RUN
      USE MITGCM_ORG_ATM, ONLY: ATM_SIZE => GET_DOMAIN_SIZE

      IMPLICIT NONE
      PRIVATE
      PUBLIC ESMF_SET_SERVICES
      CONTAINS

      SUBROUTINE ESMF_SET_SERVICES( gC, esmfRC )
!     == Routine arguments ==
      TYPE(ESMF_GridComp) :: gC
      INTEGER             :: esmfRC

!     Register init handler
      CALL ESMF_GridCompSetEntryPoint( gC, ESMF_SETINIT,
     &            driver_init, ESMF_SINGLEPHASE, esmfRC )

!     Register run handler
      CALL ESMF_GridCompSetEntryPoint( gC, ESMF_SETRUN,
     &            driver_run, ESMF_SINGLEPHASE, esmfRC )

      RETURN
      END SUBROUTINE

      SUBROUTINE DRIVER_INIT( gC, iState, eState, clock, esmfRC )

!     == Global variables  ==
!     Uses module static data EXPORT_NAMES

!     == Routine arguments ==
      TYPE(ESMF_GridComp) :: gC
      TYPE(ESMF_State)    :: iState
      TYPE(ESMF_State)    :: eState
      TYPE(ESMF_Clock)    :: clock
      INTEGER             :: esmfRC

!     == Local variables ==
!     nx, ny, nr          ## Component index space extents
      INTEGER             :: nx, ny, nr, OL

!     Query component for its size
      CALL ATM_SIZE( nx, ny, nr, OL )

!     Create and attach the fields to the import and export states
      CALL ATM_MAKE_IMPEXP_FIELDS( gC, iState, eState,
     &                             nx, ny, OL )

!     Print out summary to make sure we have what we expect
!D    CALL ESMF_StatePrint( iState, "no-opt", esmfRC )
!D    CALL ESMF_StatePrint( eState, "no-opt", esmfRC )

!     Do the component initialization
      CALL DO_ATM_INIT( iState, eState )
  
      RETURN
      END SUBROUTINE

      SUBROUTINE DRIVER_RUN(  gC, iState, eState, clock, esmfRC )

!     == Routine arguments ==
      TYPE(ESMF_GridComp) :: gC
      TYPE(ESMF_State)    :: iState
      TYPE(ESMF_State)    :: eState
      TYPE(ESMF_Clock)    :: clock
      INTEGER             :: esmfRC

!     == Local arguments ==
      TYPE(ESMF_Time)       :: currTime
      INTEGER(ESMF_KIND_I8) :: aCount

!     Write out the time period we are going to execute
      CALL ESMF_ClockGet( clock, currTime=currTime, rc=esmfRC )
      CALL ESMF_ClockGet( clock, advanceCount=aCount, rc=esmfRC )
      WRITE(0,*) 'ATM run called for period ',aCount,' to ',aCount+1,' days from start.'

      CALL DO_ATM_RUN(iState, eState, aCount )


      RETURN
      END SUBROUTINE

      SUBROUTINE ATM_MAKE_IMPEXP_FIELDS( gC, iState, eState,
     &                                   nx, ny, OL )
!     =========================================================
!     = S/R ATM_MAKE_IMPEXP_FIELDS: Creates esmf import and 
!     = export fields for an ATM component and attaches them
!     = to the components import and export states.
!     =========================================================

!     == Routine arguments ==
      TYPE(ESMF_GridComp) :: gC
      TYPE(ESMF_State)    :: iState
      TYPE(ESMF_State)    :: eState
      INTEGER             :: nx
      INTEGER             :: ny
      INTEGER             :: OL

!     == Local variables ==
      TYPE(ESMF_DELayout)  :: cLayout
      INTEGER              :: cDEnx, cDEny, cMyDEx, cMyDEy
      INTEGER              :: gridCount(2)
      REAL(ESMF_KIND_R8)   :: gridLo(2), gridHi(2)
      TYPE(ESMF_Logical)   :: periodic(2)
      INTEGER              :: haloW
      TYPE(ESMF_Grid)      :: gridRef
      TYPE(ESMF_ArraySpec) :: arraySpec
      INTEGER              :: esmfRC
      TYPE(ESMF_Field)     :: fieldRef
      INTEGER              :: I

!     Before we can create fields to attach to the import and export states
!     we need a a Grid and and Array Spec.
!     Extract layout information for the component - needed to create a grid
      CALL ESMF_GridCompGet( gC, layout=cLayout, rc=esmfRC             )
      CALL ESMF_DELayoutGetSize( cLayout, cDEnx, cDEny, esmfRC         )
      CALL ESMF_DELayoutGetDEPosition( cLayout, cMyDEx, cMyDEy, esmfRC )
!     Create grid
      gridCount(1) = nx
      gridCount(2) = ny
      gridLo(1)    = 0.
      gridLo(2)    = 0.
      gridHi(1)    = 1.
      gridHi(2)    = 1.
      haloW        = OL
      periodic(1)  = ESMF_TRUE
      periodic(2)  = ESMF_TRUE
      gridRef      = ESMF_GridCreate(2,
     &               counts=gridCount,
     &               min=gridLo,
     &               max=gridHi,
     &               layout=cLayout,
     &               horz_gridtype=ESMF_GridType_XY,
     &               horz_stagger=ESMF_GridStagger_A,
     &               horz_coord_system=ESMF_CoordSystem_Cartesian,
     &               periodic=periodic,
     &               name="atm horiz grid",
     &               rc=esmfRC)
!     Crate an array spec.
      CALL ESMF_ArraySpecInit( arraySpec, rank=2, type=ESMF_DATA_REAL,
     &                         kind=ESMF_R8)

!     b) Now create the actual fields
!     o Imports list
!     "ocn hocn"               - m (0 at seasurface, +ve upward)
!     "ocn sstocn"             - oC
      DO I=1,MAX_IMPORTS
       fieldRef = ESMF_FieldCreate( gridRef, arraySpec,
     &            relloc=ESMF_CELL_CENTER,
     &            haloWidth=haloW,
     &            name=import_names(I),
     &            rc=esmfRC)
       CALL ESMF_StateAddData(iState, fieldRef, esmfRC )
      ENDDO

!     o Exports list
!     "atm heatflux"           - W/m^2
!     "atm taux"               - N/m^2
!     "atm tauy"               - N/m^2
!     "atm latent heatflux"    - W/m^2
!     "atm sensible heatflux"  - W/m^2
!     "atm longwave heatflux"  - W/m^2
!     "atm shortwave heatflux" - W/m^2
!     "atm uvelground"         - m/s   
!     "atm vvelground"         - m/s   
!     "atm fwflux"             - m/s   
!     "atm Hatm"               - Pa
      DO I=1,MAX_EXPORTS
       fieldRef = ESMF_FieldCreate( gridRef, arraySpec,
     &            relloc=ESMF_CELL_CENTER,
     &            haloWidth=haloW,
     &            name=export_names(I),
     &           rc=esmfRC)
       CALL ESMF_StateAddData(eState, fieldRef, esmfRC )
      ENDDO

      RETURN
      END SUBROUTINE
      SUBROUTINE DO_ATM_INIT( iState, eState )
!     =========================================================
!     = S/R DO_ATM_INIT: MITgcm ATM component initliatisation
!     = driver that binds to ESMF import/export state self-describing
!     = argument based inter-component data flow abstraction.
!     = Routine (1.) takes data from ESMF import/export name space
!     = into MITgcm ATM internal data structures at start, (2.)
!     = calls MITgcm ATM internal initialization sequence and (3.)
!     = puts data from MITgcm ATM internal data structures
!     = into ESMF import/export name space prior to return.
!     =========================================================

!     == Routine arguments ==
      TYPE(ESMF_State)   :: iState
      TYPE(ESMF_State)   :: eState

!     == Local variables ==
      TYPE(ESMF_Field)   :: fieldRef
      TYPE(ESMF_Array)   :: arrayRef
      TYPE(ESMF_AxisIndex), dimension(ESMF_MAXGRIDDIM) :: indexc
      TYPE(ESMF_AxisIndex), dimension(ESMF_MAXGRIDDIM) :: indext
      INTEGER            :: esmfRC

!     Variables for communicating component for boundary state
      REAL*8, DIMENSION(:,:), ALLOCATABLE :: atm_HeatFlux
      REAL*8, DIMENSION(:,:), ALLOCATABLE :: atm_tauX
      REAL*8, DIMENSION(:,:), ALLOCATABLE :: atm_tauY
      REAL*8, DIMENSION(:,:), ALLOCATABLE :: atm_Qlatent
      REAL*8, DIMENSION(:,:), ALLOCATABLE :: atm_Qsensible
      REAL*8, DIMENSION(:,:), ALLOCATABLE :: atm_Qlongwave
      REAL*8, DIMENSION(:,:), ALLOCATABLE :: atm_Qshortwave
      REAL*8, DIMENSION(:,:), ALLOCATABLE :: atm_uVelGround
      REAL*8, DIMENSION(:,:), ALLOCATABLE :: atm_vVelGround
      REAL*8, DIMENSION(:,:), ALLOCATABLE :: atm_FWFlux
      REAL*8, DIMENSION(:,:), ALLOCATABLE :: atm_Hatm
      REAL*8, DIMENSION(:,:), ALLOCATABLE :: ocn_Hocn
      REAL*8, DIMENSION(:,:), ALLOCATABLE :: ocn_SSTocn
      INTEGER                             :: compsNx, compsNy
      INTEGER                             :: iHiC, iLoC, jHiC, jLoC
      INTEGER                             :: iHiT, OL, I, J
      REAL(KIND=ESMF_KIND_R8), DIMENSION(:,:), POINTER :: dataPtr

!
!     (1.) Take values from ESMF import state name space and put into
!          MITgcm internal data structures.
!     
!     Figure out bounds for dynamic arrays used to pass data "down" to
!     the component.
      CALL ESMF_StateGetData( eState, export_names(1), 
     &                        fieldRef, esmfRC )
      CALL ESMF_FieldGetData( fieldRef, arrayRef, esmfRC )
      CALL ESMF_ArrayGetAxisIndex( arrayRef,  compindex=indexc,
     &                                       totalindex=indext,
     &                                               rc=esmfRC)
      jLoC    = indexc(2)%min
      jHiC    = indexc(2)%max
      iLoC    = indexc(1)%min
      iHiC    = indexc(1)%max
      iHiT    = indext(1)%max
      OL      = iHiT - iHiC
      compsNx = iHiC-iLoC+1
      compsNy = jHiC-jLoC+1
!     Allocate dynamic arrays
      ALLOCATE( atm_HeatFlux(      1-OL:compsNx+OL, 1-OL:compsNy+OL) )
      ALLOCATE( atm_TauX(          1-OL:compsNx+OL, 1-OL:compsNy+OL) )
      ALLOCATE( atm_TauY(          1-OL:compsNx+OL, 1-OL:compsNy+OL) )
      ALLOCATE( atm_Qlatent(       1-OL:compsNx+OL, 1-OL:compsNy+OL) )
      ALLOCATE( atm_Qsensible(     1-OL:compsNx+OL, 1-OL:compsNy+OL) )
      ALLOCATE( atm_Qlongwave(     1-OL:compsNx+OL, 1-OL:compsNy+OL) )
      ALLOCATE( atm_Qshortwave(    1-OL:compsNx+OL, 1-OL:compsNy+OL) )
      ALLOCATE( atm_uVelground(    1-OL:compsNx+OL, 1-OL:compsNy+OL) )
      ALLOCATE( atm_vVelground(    1-OL:compsNx+OL, 1-OL:compsNy+OL) )
      ALLOCATE( atm_FWFlux(        1-OL:compsNx+OL, 1-OL:compsNy+OL) )
      ALLOCATE( atm_Hatm(          1-OL:compsNx+OL, 1-OL:compsNy+OL) )
      ALLOCATE( ocn_Hocn(          1-OL:compsNx+OL, 1-OL:compsNy+OL) )
      ALLOCATE( ocn_SSTocn(        1-OL:compsNx+OL, 1-OL:compsNy+OL) )

!
!     (2.) Execute MITgcm ATM internal initialization sequence.
!
      CALL ATM_INIT(
     O              atm_HeatFlux, atm_TauX, atm_TauY,
     O              atm_Qlatent, atm_Qsensible, atm_Qlongwave,
     O              atm_Qshortwave,
     O              atm_uVelGround, atm_vVelGround,
     O              atm_FWFlux, atm_Hatm,
     I              ocn_Hocn, ocn_SSTocn
     &             )

!
!     (3.) Extract values from MITgcm internal data structures and 
!          place them into ESMF export state name space.
!
!     mitgcm_org_atm heatflux
      CALL STATE_GET_FADP( eState, export_names( 1), dataPtr )
      DO J=1,compsNy
       DO I=1, compsNx
        dataPtr(iLoC+i-1,jLoc+j-1) = atm_heatflux(i,j)
       ENDDO
      ENDDO
!     mitgcm_org_atm taux
      CALL STATE_GET_FADP( eState, export_names( 2), dataPtr )
      DO J=1,compsNy
       DO I=1, compsNx
        dataPtr(iLoC+i-1,jLoc+j-1) = atm_taux(i,j)
       ENDDO
      ENDDO
!     mitgcm_org_atm tauy
      CALL STATE_GET_FADP( eState, export_names( 3), dataPtr )
      DO J=1,compsNy
       DO I=1, compsNx
        dataPtr(iLoC+i-1,jLoc+j-1) = atm_tauy(i,j)
       ENDDO
      ENDDO
!     mitgcm_org_atm latent heatflux
      CALL STATE_GET_FADP( eState, export_names( 4), dataPtr )
      DO J=1,compsNy
       DO I=1, compsNx
        dataPtr(iLoC+i-1,jLoc+j-1) = atm_qlatent(i,j)
       ENDDO
      ENDDO
!     mitgcm_org_atm sensible heatflux
      CALL STATE_GET_FADP( eState, export_names( 5), dataPtr )
      DO J=1,compsNy
       DO I=1, compsNx
        dataPtr(iLoC+i-1,jLoc+j-1) = atm_qsensible(i,j)
       ENDDO
      ENDDO
!     mitgcm_org_atm longwave heatflux
      CALL STATE_GET_FADP( eState, export_names( 6), dataPtr )
      DO J=1,compsNy
       DO I=1, compsNx
        dataPtr(iLoC+i-1,jLoc+j-1) = atm_qlongwave(i,j)
       ENDDO
      ENDDO
!     mitgcm_org_atm shortwave heatflux
      CALL STATE_GET_FADP( eState, export_names( 7), dataPtr )
      DO J=1,compsNy
       DO I=1, compsNx
        dataPtr(iLoC+i-1,jLoc+j-1) = atm_qshortwave(i,j)
       ENDDO
      ENDDO
!     mitgcm_org_atm uvelground
      CALL STATE_GET_FADP( eState, export_names( 8), dataPtr )
      DO J=1,compsNy
       DO I=1, compsNx
        dataPtr(iLoC+i-1,jLoc+j-1) = atm_uvelground(i,j)
       ENDDO
      ENDDO
!     mitgcm_org_atm vvelground
      CALL STATE_GET_FADP( eState, export_names( 9), dataPtr )
      DO J=1,compsNy
       DO I=1, compsNx
        dataPtr(iLoC+i-1,jLoc+j-1) = atm_vvelground(i,j)
       ENDDO
      ENDDO
!     mitgcm_org_atm fwflux
      CALL STATE_GET_FADP( eState, export_names(10), dataPtr )
      DO J=1,compsNy
       DO I=1, compsNx
        dataPtr(iLoC+i-1,jLoc+j-1) = atm_fwflux(i,j)
       ENDDO
      ENDDO
!     mitgcm_org_atm hatm
      CALL STATE_GET_FADP( eState, export_names(11), dataPtr )
      DO J=1,compsNy
       DO I=1, compsNx
        dataPtr(iLoC+i-1,jLoc+j-1) = atm_hatm(i,j)
       ENDDO
      ENDDO

!     Clean up
      DEALLOCATE( atm_HeatFlux   )
      DEALLOCATE( atm_TauX       )
      DEALLOCATE( atm_TauY       )
      DEALLOCATE( atm_Qlatent    )
      DEALLOCATE( atm_Qsensible  )
      DEALLOCATE( atm_Qlongwave  )
      DEALLOCATE( atm_Qshortwave )
      DEALLOCATE( atm_uVelground )
      DEALLOCATE( atm_vVelground )
      DEALLOCATE( atm_FWFlux     )
      DEALLOCATE( ocn_Hocn       )
      DEALLOCATE( ocn_SSTocn     )

      RETURN
      END SUBROUTINE
      SUBROUTINE DO_ATM_RUN(  iState, eState, aCount )
!     =========================================================
!     = S/R DO_ATM_RUN: MITgcm ATM component runstep execution
!     = driver that binds to ESMF import/export state self-describing
!     = argument based inter-component data flow abstraction.
!     = Routine (1.) takes data from ESMF import/export name space
!     = into MITgcm ATM internal data structures at start, (2.)
!     = calls MITgcm ATM internal run step sequence and (3.)
!     = puts data from MITgcm ATM internal data structures
!     = into ESMF import/export name space prior to return.
!     =========================================================

!     == Routine arguments ==
      TYPE(ESMF_State)      :: iState
      TYPE(ESMF_State)      :: eState
      INTEGER(ESMF_KIND_I8) :: aCount

!     == Local variables ==
      TYPE(FIARDA)       :: fS
!     Variables for communication compoent boundary state
!     data in MITgcm ATM import state
      REAL*8, DIMENSION(:,:), ALLOCATABLE :: ocn_SSTocn
      REAL*8, DIMENSION(:,:), ALLOCATABLE :: ocn_Hocn
!     data in MITgcm ATM export state
      REAL*8, DIMENSION(:,:), ALLOCATABLE :: atm_HeatFlux
      REAL*8, DIMENSION(:,:), ALLOCATABLE :: atm_tauX
      REAL*8, DIMENSION(:,:), ALLOCATABLE :: atm_tauY
      REAL*8, DIMENSION(:,:), ALLOCATABLE :: atm_Qlatent
      REAL*8, DIMENSION(:,:), ALLOCATABLE :: atm_Qsensible
      REAL*8, DIMENSION(:,:), ALLOCATABLE :: atm_Qlongwave
      REAL*8, DIMENSION(:,:), ALLOCATABLE :: atm_Qshortwave
      REAL*8, DIMENSION(:,:), ALLOCATABLE :: atm_uVelGround
      REAL*8, DIMENSION(:,:), ALLOCATABLE :: atm_vVelGround
      REAL*8, DIMENSION(:,:), ALLOCATABLE :: atm_FWFlux
      REAL*8, DIMENSION(:,:), ALLOCATABLE :: atm_Hatm
!     data pointer and variables used in extract data from 
!     ESMF state objects
      REAL(KIND=ESMF_KIND_R8), DIMENSION(:,:), POINTER :: dP
      INTEGER :: OL, cNx, cNy, I, J
      TYPE(ESMF_Field)                    :: fieldRef
      INTEGER :: esmfRC, startStep, stopStep
!
!     (1.) Determine size and bounds of work arrays and allocate space
      CALL STATE_GET_FADP( iState, import_names(1), dP, fSpec=fS )
!     Allocate dynamic work arrays
      OL  = fS%hW
      cNx = fS%nI(1)
      cNy = fS%nI(2)
      ALLOCATE( ocn_SSTocn(        1-OL:cNx+OL, 1-OL:cNy+OL) )
      ALLOCATE( ocn_Hocn(          1-OL:cNx+OL, 1-OL:cNy+OL) )
      ALLOCATE( atm_HeatFlux(      1-OL:cNx+OL, 1-OL:cNy+OL) )
      ALLOCATE( atm_TauX(          1-OL:cNx+OL, 1-OL:cNy+OL) )
      ALLOCATE( atm_TauY(          1-OL:cNx+OL, 1-OL:cNy+OL) )
      ALLOCATE( atm_Qlatent(       1-OL:cNx+OL, 1-OL:cNy+OL) )
      ALLOCATE( atm_Qsensible(     1-OL:cNx+OL, 1-OL:cNy+OL) )
      ALLOCATE( atm_Qlongwave(     1-OL:cNx+OL, 1-OL:cNy+OL) )
      ALLOCATE( atm_Qshortwave(    1-OL:cNx+OL, 1-OL:cNy+OL) )
      ALLOCATE( atm_uVelground(    1-OL:cNx+OL, 1-OL:cNy+OL) )
      ALLOCATE( atm_vVelground(    1-OL:cNx+OL, 1-OL:cNy+OL) )
      ALLOCATE( atm_FWFlux(        1-OL:cNx+OL, 1-OL:cNy+OL) )
      ALLOCATE( atm_Hatm(          1-OL:cNx+OL, 1-OL:cNy+OL) )

!     (2.) Extract import state (including overlap region)
      CALL STATE_GET_FADP( iState, import_names( 1), dP)
      DO J=1-OL,cNy+OL
       DO I=1-OL,cNx+OL
        ocn_Hocn(i,j) = dP(fS%LoC(1)+i-1,fS%LoC(2)+j-1)
       ENDDO
      ENDDO
      CALL STATE_GET_FADP( iState, import_names( 2), dP)
      DO J=1-OL,cNy+OL
       DO I=1-OL,cNx+OL
        ocn_SSTocn(i,j) = dP(fS%LoC(1)+i-1,fS%LoC(2)+j-1)
       ENDDO
      ENDDO

!     (3.) Execute run step
      startStep = aCount*192
      stopStep  = startStep+192
      CALL ATM_RUN(atm_HeatFlux, atm_TauX, atm_TauY,
     O             atm_Qlatent, atm_Qsensible, atm_Qlongwave,
     O             atm_Qshortwave,
     O             atm_uVelGround, atm_vVelGround,
     O             atm_FWFlux,
     I             ocn_Hocn, ocn_SSTocn,
     I             startStep, stopStep
     &                )

!     mitgcm_org_atm heatflux
      CALL STATE_GET_FADP( eState, export_names( 1), dP , theFieldRef=fieldRef )
      DO J=1,cNy
       DO I=1,cNx
        dP(fS%LoC(1)+i-1,fS%LoC(2)+j-1) = atm_Heatflux(i,j)
       ENDDO
      ENDDO
      CALL ESMF_FieldHalo( fieldRef )
!     mitgcm_org_atm taux
      CALL STATE_GET_FADP( eState, export_names( 2), dP , theFieldRef=fieldRef )
      DO J=1,cNy
       DO I=1,cNx
        dP(fS%LoC(1)+i-1,fS%LoC(2)+j-1) = atm_TauX(i,j)
       ENDDO
      ENDDO
      CALL ESMF_FieldHalo( fieldRef )
!     mitgcm_org_atm tauy
      CALL STATE_GET_FADP( eState, export_names( 3), dP , theFieldRef=fieldRef )
      DO J=1,cNy
       DO I=1,cNx
        dP(fS%LoC(1)+i-1,fS%LoC(2)+j-1) = atm_TauY(i,j)
       ENDDO
      ENDDO
      CALL ESMF_FieldHalo( fieldRef )
!     mitgcm_org_atm latent heatflux
      CALL STATE_GET_FADP( eState, export_names( 4), dP , theFieldRef=fieldRef )
      DO J=1,cNy
       DO I=1,cNx
        dP(fS%LoC(1)+i-1,fS%LoC(2)+j-1) = atm_Qlatent(i,j)
       ENDDO
      ENDDO
      CALL ESMF_FieldHalo( fieldRef )
!     mitgcm_org_atm sensible heatflux
      CALL STATE_GET_FADP( eState, export_names( 5), dP , theFieldRef=fieldRef )
      DO J=1,cNy
       DO I=1,cNx
        dP(fS%LoC(1)+i-1,fS%LoC(2)+j-1) = atm_Qsensible(i,j)
       ENDDO
      ENDDO
      CALL ESMF_FieldHalo( fieldRef )
!     mitgcm_org_atm longwave heatflux
      CALL STATE_GET_FADP( eState, export_names( 6), dP , theFieldRef=fieldRef )
      DO J=1,cNy
       DO I=1,cNx
        dP(fS%LoC(1)+i-1,fS%LoC(2)+j-1) = atm_Qlongwave(i,j)
       ENDDO
      ENDDO
      CALL ESMF_FieldHalo( fieldRef )
!     mitgcm_org_atm shortwave heatflux
      CALL STATE_GET_FADP( eState, export_names( 7), dP , theFieldRef=fieldRef )
      DO J=1,cNy
       DO I=1,cNx
        dP(fS%LoC(1)+i-1,fS%LoC(2)+j-1) = atm_Qshortwave(i,j)
       ENDDO
      ENDDO
      CALL ESMF_FieldHalo( fieldRef )
!     mitgcm_org_atm uvelground
      CALL STATE_GET_FADP( eState, export_names( 8), dP , theFieldRef=fieldRef )
      DO J=1,cNy
       DO I=1,cNx
        dP(fS%LoC(1)+i-1,fS%LoC(2)+j-1) = atm_uVelground(i,j)
       ENDDO
      ENDDO
      CALL ESMF_FieldHalo( fieldRef )
!     mitgcm_org_atm vvelground
      CALL STATE_GET_FADP( eState, export_names( 9), dP , theFieldRef=fieldRef )
      DO J=1,cNy
       DO I=1,cNx
        dP(fS%LoC(1)+i-1,fS%LoC(2)+j-1) = atm_vVelground(i,j)
       ENDDO
      ENDDO
      CALL ESMF_FieldHalo( fieldRef )
!     mitgcm_org_atm fwflux
      CALL STATE_GET_FADP( eState, export_names(10), dP , theFieldRef=fieldRef )
      DO J=1,cNy
       DO I=1,cNx
        dP(fS%LoC(1)+i-1,fS%LoC(2)+j-1) = atm_FWFlux(i,j)
       ENDDO
      ENDDO
      CALL ESMF_FieldHalo( fieldRef )

!     (4.) Transfer updated outputs to export state space and
!          deallocate work space.
      DEALLOCATE( ocn_SSTocn     )
      DEALLOCATE( ocn_Hocn       )
      DEALLOCATE( atm_HeatFlux   )
      DEALLOCATE( atm_TauX       )
      DEALLOCATE( atm_TauY       )
      DEALLOCATE( atm_Qlatent    )
      DEALLOCATE( atm_Qsensible  )
      DEALLOCATE( atm_Qlongwave  )
      DEALLOCATE( atm_Qshortwave )
      DEALLOCATE( atm_uVelground )
      DEALLOCATE( atm_vVelground )
      DEALLOCATE( atm_FWFlux     )
      DEALLOCATE( atm_Hatm       )

      RETURN
      END SUBROUTINE
      END MODULE
