! $Header: $
! $Name:   $
      MODULE MITGCM_ORG_OCN_ESMF_DRIVER

      USE ESMF_MOD
      USE MITGCM_ORG_ESMF_UTILS
      USE MITGCM_ORG_ESMF_EXCH
      USE MITGCM_ORG_OCN_ESMF_IMPORTS
      USE MITGCM_ORG_OCN_ESMF_EXPORTS

      USE MITGCM_ORG_OCN, ONLY: OCN_INIT => DRIVER_INIT
      USE MITGCM_ORG_OCN, ONLY: OCN_RUN  => DRIVER_RUN
      USE MITGCM_ORG_OCN, ONLY: OCN_SIZE => GET_DOMAIN_SIZE

      IMPLICIT NONE

      PRIVATE
      PUBLIC ESMF_SET_SERVICES

      CONTAINS

      SUBROUTINE ESMF_SET_SERVICES( gC, esmfRC )
!     == Routine arguments ==
      TYPE(ESMF_GridComp) :: gC
      INTEGER             :: esmfRC

!     Register init handler
      CALL ESMF_GridCompSetEntryPoint( gC, ESMF_SETINIT,
     &            driver_init, ESMF_SINGLEPHASE, esmfRC )

!     Register run handler
      CALL ESMF_GridCompSetEntryPoint( gC, ESMF_SETRUN,
     &            driver_run, ESMF_SINGLEPHASE, esmfRC )

      RETURN
      END SUBROUTINE

      SUBROUTINE DRIVER_INIT( gC, iState, eState, clock, esmfRC )

!     == Global variables  ==
      USE MITGCM_ORG_ESMF_GS

!     == Routine arguments ==
      TYPE(ESMF_GridComp) :: gC
      TYPE(ESMF_State)    :: iState
      TYPE(ESMF_State)    :: eState
      TYPE(ESMF_Clock)    :: clock
      INTEGER             :: esmfRC

!     == Local variables ==
!     nx, ny, nr          ## Component index space extents
      INTEGER             :: nx, ny, nr, OL

!     Query component for its size
      CALL OCN_SIZE( nx, ny, nr, OL )

!     Setup communication handlers
      CALL GLOBAL_SUM_E_INIT( gC           )
      CALL EXCH_E_INIT( gC, nx, ny, nr, OL )

!     Create and attach the fields to the import and export states
      CALL OCN_MAKE_IMPEXP_FIELDS( gC, iState, eState,
     &                             nx, ny, OL )

!     Print out summary to make sure we have what we expected
      CALL ESMF_StatePrint( iState, "no-opt", esmfRC )
      CALL ESMF_StatePrint( eState, "no-opt", esmfRC )

!     Now call component internal initialization
      CALL DO_OCN_INIT( iState, eState )

      RETURN
      END SUBROUTINE

      SUBROUTINE DRIVER_RUN(  gC, iState, eState, clock, esmfRC )

!     == Routine arguments ==
      TYPE(ESMF_GridComp) :: gC
      TYPE(ESMF_State)    :: iState
      TYPE(ESMF_State)    :: eState
      TYPE(ESMF_Clock)    :: clock
      INTEGER             :: esmfRC
      REAL(KIND=ESMF_KIND_R8) :: days

!     == Local arguments ==
      TYPE(ESMF_Time)       :: currTime
      INTEGER(ESMF_KIND_I8) :: aCount
!     Variables for creating an array specification

!     Write out the time period we are going to execute
      CALL ESMF_ClockGet( clock, currTime=currTime, rc=esmfRC )
      CALL ESMF_ClockGet( clock, advanceCount=aCount, rc=esmfRC )
      WRITE(0,*) 'OCN run called for period ',aCount,' to ',aCount+1,' days from start.'

      CALL DO_OCN_RUN( iState, eState, aCount )

      RETURN
      END SUBROUTINE
      SUBROUTINE OCN_MAKE_IMPEXP_FIELDS( gC, iState, eState, nx, ny, OL )
!     =========================================================
!     = S/R OCN_MAKE_IMPEXP_FIELDS: Creates esmf import and
!     = export fields for an OCN component and attaches them
!     = to the components import and export states.
!     =========================================================

!     == Routine arguments ==
      TYPE(ESMF_GridComp) :: gC
      TYPE(ESMF_State)    :: iState
      TYPE(ESMF_State)    :: eState
      INTEGER             :: nx
      INTEGER             :: ny
      INTEGER             :: OL

!     == Local variables ==
      TYPE(ESMF_DELayout)  :: cLayout
      INTEGER              :: esmfRC
      INTEGER              :: cMyDEx, cMyDEy, cDEnx, cDEny
      INTEGER              :: gridCount(2)
      REAL(ESMF_KIND_R8)   :: gridLo(2), gridHi(2)
      TYPE(ESMF_Logical)   :: periodic(2)
      INTEGER              :: haloW
      TYPE(ESMF_Grid)      :: gridRef
      TYPE(ESMF_ArraySpec) :: arraySpec
      TYPE(ESMF_Field)     :: fieldRef
      INTEGER              :: I

!     Extract layout information for the component
      CALL ESMF_GridCompGet( gC, layout=cLayout, rc=esmfRC             )
      CALL ESMF_DELayoutGetSize( cLayout, cDEnx, cDEny, esmfRC         )
      CALL ESMF_DELayoutGetDEPosition( cLayout, cMyDEx, cMyDEy, esmfRC )

!     Create the fields for use in import and export state.
!     a) First we need a grid and an array spec, the we can create 
!        fields and put refences to the fields in iState and eState.
!        For MITgcm OCN ESMF driver grid is 2d with same size as OCN 
!        component horizontal extents and doubly periodic. 
!        o grid
      gridCount(1) = nx
      gridCount(2) = ny
      gridLo(1)    = 0.
      gridLo(2)    = 0.
      gridHi(1)    = 1.
      gridHi(2)    = 1.
      haloW        = OL
      periodic(1)  = ESMF_TRUE
      periodic(2)  = ESMF_TRUE
      gridRef      = ESMF_GridCreate(2,
     &               counts=gridCount,
     &               min=gridLo,
     &               max=gridHi,
     &               layout=cLayout,
     &               horz_gridtype=ESMF_GridType_XY,
     &               horz_stagger=ESMF_GridStagger_A,
     &               horz_coord_system=ESMF_CoordSystem_Cartesian,
     &               periodic=periodic,
     &               name="ocn horiz grid",
     &               rc=esmfRC)
!        o array spec
      CALL ESMF_ArraySpecInit( arraySpec, rank=2, type=ESMF_DATA_REAL,
     &                         kind=ESMF_R8)

!     b) Now create the actual fields
!     o Imports list
!     "atm heatflux"           - W/m^2
!     "atm taux"               - N/m^2
!     "atm tauy"               - N/m^2
!     "atm latent heatflux"    - W/m^2
!     "atm sensible heatflux"  - W/m^2
!     "atm longwave heatflux"  - W/m^2
!     "atm shortwave heatflux" - W/m^2
!     "atm uvelground"         - m/s
!     "atm vvelground"         - m/s
!     "atm fwflux"             - m/s
!     "atm Hatm"               - Pa
      DO I=1,MAX_IMPORTS
       fieldRef = ESMF_FieldCreate( gridRef, arraySpec,
     &            relloc=ESMF_CELL_CENTER,
     &            haloWidth=haloW,
     &            name=import_names(I),
     &            rc=esmfRC)
       CALL ESMF_StateAddData(iState, fieldRef, esmfRC )
      ENDDO
!     o exports list
!     "ocn hocn"           - m (0 at seasurface, +ve upward)
!     "ocn sstocn"         - oC
      DO I=1,MAX_EXPORTS
       fieldRef = ESMF_FieldCreate( gridRef, arraySpec,
     &            relloc=ESMF_CELL_CENTER,
     &            haloWidth=haloW,
     &            name=export_names(I),
     &            rc=esmfRC)
       CALL ESMF_StateAddData(eState, fieldRef, esmfRC )
      ENDDO

      RETURN
      END SUBROUTINE
      SUBROUTINE DO_OCN_INIT( iState, eState )
!     =========================================================
!     = S/R DO_OCN_INIT: MITgcm OCN component initliatisation
!     = driver that binds to ESMF import/export state self-describing
!     = argument based inter-component data flow abstraction.
!     = Routine (1.) takes data from ESMF import/export name space
!     = into MITgcm OCN internal data structures at start, (2.)
!     = calls MITgcm OCN internal initialization sequence and (3.)
!     = puts data from MITgcm OCN internal data structures
!     = into ESMF import/export name space prior to return.
!     =========================================================

!     == Routine arguments ==
      TYPE(ESMF_State)   :: iState
      TYPE(ESMF_State)   :: eState

!     == Local variables ==
      TYPE(ESMF_Field)   :: fieldRef
      TYPE(ESMF_Array)   :: arrayRef
      TYPE(ESMF_AxisIndex), dimension(ESMF_MAXGRIDDIM) :: indexc
      TYPE(ESMF_AxisIndex), dimension(ESMF_MAXGRIDDIM) :: indext
      INTEGER            :: esmfRC

!     Variables for communicating component for boundary state
      REAL*8, DIMENSION(:,:), ALLOCATABLE :: atm_HeatFlux
      REAL*8, DIMENSION(:,:), ALLOCATABLE :: atm_tauX
      REAL*8, DIMENSION(:,:), ALLOCATABLE :: atm_tauY
      REAL*8, DIMENSION(:,:), ALLOCATABLE :: atm_Qlatent
      REAL*8, DIMENSION(:,:), ALLOCATABLE :: atm_Qsensible
      REAL*8, DIMENSION(:,:), ALLOCATABLE :: atm_Qlongwave
      REAL*8, DIMENSION(:,:), ALLOCATABLE :: atm_Qshortwave
      REAL*8, DIMENSION(:,:), ALLOCATABLE :: atm_uVelGround
      REAL*8, DIMENSION(:,:), ALLOCATABLE :: atm_vVelGround
      REAL*8, DIMENSION(:,:), ALLOCATABLE :: atm_FWFlux
      REAL*8, DIMENSION(:,:), ALLOCATABLE :: atm_Hatm
      REAL*8, DIMENSION(:,:), ALLOCATABLE :: ocn_Hocn
      REAL*8, DIMENSION(:,:), ALLOCATABLE :: ocn_SSTocn
      INTEGER                             :: compsNx, compsNy
      INTEGER                             :: iHiC, iLoC, jHiC, jLoC
      INTEGER                             :: iHiT, OL, I, J
      REAL(KIND=ESMF_KIND_R8), DIMENSION(:,:), POINTER :: dataPtr

!     Determine size and bounds of work arrays
      CALL ESMF_StateGetData( eState, export_names(1), fieldRef, esmfRC )
      CALL ESMF_FieldGetData( fieldRef, arrayRef, esmfRC)
      CALL ESMF_ArrayGetAxisIndex( arrayRef, totalindex=indext,
     &                                        compindex=indexc,
     &                                               rc=esmfRC)
      CALL ESMF_ArrayGetData( arrayRef, dataPtr, ESMF_DATA_REF, esmfRC)
!     Array (j[i]Lo[Hi]C interior bounds, j[i]Lo[Hi]T total bounds)
!     Bounds are indexed from 1 and sub-divided according to DE layout.
!     Index 1 corresponds to edge of overlap/ghost/halo region.
      jLoC = indexc(2)%min
      jHiC = indexc(2)%max
      iLoC = indexc(1)%min
      iHiC = indexc(1)%max
      iHiT = indext(1)%max
      OL      = iHiT-iHiC
      compsNx = iHiC-iLoC+1
      compsNy = jHiC-jLoC+1
!     Allocate dynamic work arrays
      ALLOCATE( atm_HeatFlux(      1-OL:compsNx+OL, 1-OL:compsNy+OL) )
      ALLOCATE( atm_TauX(          1-OL:compsNx+OL, 1-OL:compsNy+OL) )
      ALLOCATE( atm_TauY(          1-OL:compsNx+OL, 1-OL:compsNy+OL) )
      ALLOCATE( atm_Qlatent(       1-OL:compsNx+OL, 1-OL:compsNy+OL) )
      ALLOCATE( atm_Qsensible(     1-OL:compsNx+OL, 1-OL:compsNy+OL) )
      ALLOCATE( atm_Qlongwave(     1-OL:compsNx+OL, 1-OL:compsNy+OL) )
      ALLOCATE( atm_Qshortwave(    1-OL:compsNx+OL, 1-OL:compsNy+OL) )
      ALLOCATE( atm_uVelground(    1-OL:compsNx+OL, 1-OL:compsNy+OL) )
      ALLOCATE( atm_vVelground(    1-OL:compsNx+OL, 1-OL:compsNy+OL) )
      ALLOCATE( atm_FWFlux(        1-OL:compsNx+OL, 1-OL:compsNy+OL) )
      ALLOCATE( atm_Hatm(          1-OL:compsNx+OL, 1-OL:compsNy+OL) )
      ALLOCATE( ocn_Hocn(          1-OL:compsNx+OL, 1-OL:compsNy+OL) )
      ALLOCATE( ocn_SSTocn(        1-OL:compsNx+OL, 1-OL:compsNy+OL) )

!
!     (2.) Execute MITgcm OCN internal initialization sequence.
!
      CALL OCN_INIT(
     I              atm_HeatFlux, atm_TauX, atm_TauY,
     I              atm_Qlatent, atm_Qsensible, atm_Qlongwave,
     I              atm_Qshortwave,
     I              atm_uVelGround, atm_vVelGround,
     I              atm_FWFlux,
     I              atm_Hatm,
     O              ocn_SSTocn, ocn_Hocn
     &             )

!     mitgcm_org_ocn hocn
      CALL STATE_GET_FADP( eState, export_names(1), dataPtr , theFieldRef=fieldRef )
      DO J=1,compsNy
       DO I=1, compsNx
        dataPtr(iLoC+i-1,jLoc+j-1) = ocn_Hocn(i,j)
       ENDDO
      ENDDO
      CALL ESMF_FieldHalo( fieldRef )
!     mitgcm_org_ocn sstocn  
      CALL STATE_GET_FADP( eState, export_names(2), dataPtr , theFieldRef=fieldRef )
      DO J=1,compsNy
       DO I=1, compsNx
        dataPtr(iLoC+i-1,jLoc+j-1) = ocn_SSTocn(i,j)
       ENDDO
      ENDDO
      CALL ESMF_FieldHalo( fieldRef )

!     Clean up temporaries
      DEALLOCATE( atm_HeatFlux   )
      DEALLOCATE( atm_TauX       )
      DEALLOCATE( atm_TauY       )
      DEALLOCATE( atm_Qlatent    )
      DEALLOCATE( atm_Qsensible  )
      DEALLOCATE( atm_Qlongwave  )
      DEALLOCATE( atm_Qshortwave )
      DEALLOCATE( atm_uVelground )
      DEALLOCATE( atm_vVelground )
      DEALLOCATE( atm_FWFlux     )
      DEALLOCATE( atm_Hatm       )
      DEALLOCATE( ocn_SSTocn     )
      DEALLOCATE( ocn_Hocn       )
      RETURN
      END SUBROUTINE
      SUBROUTINE DO_OCN_RUN(  iState, eState, aCount )
!     =========================================================
!     = S/R DO_OCN_RUN: MITgcm OCN component runstep execution
!     = driver that binds to ESMF import/export state self-describing
!     = argument based inter-component data flow abstraction.
!     = Routine (1.) takes data from ESMF import/export name space
!     = into MITgcm OCN internal data structures at start, (2.)
!     = calls MITgcm OCN internal run step sequence and (3.)
!     = puts data from MITgcm OCN internal data structures
!     = into ESMF import/export name space prior to return.
!     =========================================================

!     == Routine arguments ==
      TYPE(ESMF_State)      :: iState
      TYPE(ESMF_State)      :: eState
      INTEGER(ESMF_KIND_I8) :: aCount

!     == Local variables ==
      TYPE(FIARDA)       :: fS
!     Variables for communicating component for boundary state
!     data in MITgcm OCN import state
      REAL*8, DIMENSION(:,:), ALLOCATABLE :: atm_HeatFlux
      REAL*8, DIMENSION(:,:), ALLOCATABLE :: atm_tauX
      REAL*8, DIMENSION(:,:), ALLOCATABLE :: atm_tauY
      REAL*8, DIMENSION(:,:), ALLOCATABLE :: atm_Qlatent
      REAL*8, DIMENSION(:,:), ALLOCATABLE :: atm_Qsensible
      REAL*8, DIMENSION(:,:), ALLOCATABLE :: atm_Qlongwave
      REAL*8, DIMENSION(:,:), ALLOCATABLE :: atm_Qshortwave
      REAL*8, DIMENSION(:,:), ALLOCATABLE :: atm_uVelGround
      REAL*8, DIMENSION(:,:), ALLOCATABLE :: atm_vVelGround
      REAL*8, DIMENSION(:,:), ALLOCATABLE :: atm_FWFlux
      REAL*8, DIMENSION(:,:), ALLOCATABLE :: atm_Hatm
!     data in MITgcm OCN export state
      REAL*8, DIMENSION(:,:), ALLOCATABLE :: ocn_Hocn
      REAL*8, DIMENSION(:,:), ALLOCATABLE :: ocn_SSTocn
!     data pointer used in extract data from ESMF state objects
      REAL(KIND=ESMF_KIND_R8), DIMENSION(:,:), POINTER :: dP
      INTEGER :: OL, cNx, cNy, I, J
      TYPE(ESMF_Field)                    :: fieldRef
      INTEGER :: esmfRC
      INTEGER :: startStep, stopStep

!
!     Determine size and bounds of work arrays
      CALL STATE_GET_FADP( iState, import_names(1), dP, fSpec=fS )
!     Allocate dynamic work arrays
      OL  = fS%hW
      cNx = fS%nI(1)
      cNy = fS%nI(2)
      ALLOCATE( atm_HeatFlux(      1-OL:cNx+OL, 1-OL:cNy+OL) )
      ALLOCATE( atm_TauX(          1-OL:cNx+OL, 1-OL:cNy+OL) )
      ALLOCATE( atm_TauY(          1-OL:cNx+OL, 1-OL:cNy+OL) )
      ALLOCATE( atm_Qlatent(       1-OL:cNx+OL, 1-OL:cNy+OL) )
      ALLOCATE( atm_Qsensible(     1-OL:cNx+OL, 1-OL:cNy+OL) )
      ALLOCATE( atm_Qlongwave(     1-OL:cNx+OL, 1-OL:cNy+OL) )
      ALLOCATE( atm_Qshortwave(    1-OL:cNx+OL, 1-OL:cNy+OL) )
      ALLOCATE( atm_uVelground(    1-OL:cNx+OL, 1-OL:cNy+OL) )
      ALLOCATE( atm_vVelground(    1-OL:cNx+OL, 1-OL:cNy+OL) )
      ALLOCATE( atm_FWFlux(        1-OL:cNx+OL, 1-OL:cNy+OL) )
      ALLOCATE( ocn_SSTocn(        1-OL:cNx+OL, 1-OL:cNy+OL) )

!     (2.) Extract import state (including overlap region)
      CALL STATE_GET_FADP( iState, import_names( 1), dP , theFieldRef=fieldRef )
      DO J=1-OL,cNy+OL
       DO I=1-OL, cNx+OL
        atm_HeatFlux(i,j) = dP(fS%LoC(1)+i-1,fS%LoC(2)+j-1)
       ENDDO
      ENDDO
      CALL STATE_GET_FADP( iState, import_names( 2), dP , theFieldRef=fieldRef )
      DO J=1-OL,cNy+OL
       DO I=1-OL, cNx+OL
        atm_TauX(i,j) = dP(fS%LoC(1)+i-1,fS%LoC(2)+j-1)
       ENDDO
      ENDDO
      CALL STATE_GET_FADP( iState, import_names( 3), dP , theFieldRef=fieldRef )
      DO J=1-OL,cNy+OL
       DO I=1-OL, cNx+OL
        atm_TauY(i,j)      = dP(fS%LoC(1)+i-1,fS%LoC(2)+j-1)
       ENDDO
      ENDDO
      CALL STATE_GET_FADP( iState, import_names( 4), dP , theFieldRef=fieldRef )
      DO J=1-OL,cNy+OL
       DO I=1-OL, cNx+OL
        atm_Qlatent(i,j)   = dP(fS%LoC(1)+i-1,fS%LoC(2)+j-1)
       ENDDO
      ENDDO
      CALL STATE_GET_FADP( iState, import_names( 5), dP , theFieldRef=fieldRef )
      DO J=1-OL,cNy+OL
       DO I=1-OL, cNx+OL
        atm_Qsensible(i,j)  = dP(fS%LoC(1)+i-1,fS%LoC(2)+j-1)
       ENDDO
      ENDDO
      CALL STATE_GET_FADP( iState, import_names( 6), dP , theFieldRef=fieldRef )
      DO J=1-OL,cNy+OL
       DO I=1-OL, cNx+OL
        atm_Qlongwave(i,j)  = dP(fS%LoC(1)+i-1,fS%LoC(2)+j-1)
       ENDDO
      ENDDO
      CALL STATE_GET_FADP( iState, import_names( 7), dP , theFieldRef=fieldRef )
      DO J=1-OL,cNy+OL
       DO I=1-OL, cNx+OL
        atm_Qshortwave(i,j) = dP(fS%LoC(1)+i-1,fS%LoC(2)+j-1)
       ENDDO
      ENDDO
      CALL STATE_GET_FADP( iState, import_names( 8), dP , theFieldRef=fieldRef )
      DO J=1-OL,cNy+OL
       DO I=1-OL, cNx+OL
        atm_uVelGround(i,j) = dP(fS%LoC(1)+i-1,fS%LoC(2)+j-1)
       ENDDO
      ENDDO
      CALL STATE_GET_FADP( iState, import_names( 9), dP , theFieldRef=fieldRef )
      DO J=1-OL,cNy+OL
       DO I=1-OL, cNx+OL
        atm_vVelGround(i,j) = dP(fS%LoC(1)+i-1,fS%LoC(2)+j-1)
       ENDDO
      ENDDO
      CALL STATE_GET_FADP( iState, import_names(10), dP , theFieldRef=fieldRef )
      DO J=1-OL,cNy+OL
       DO I=1-OL, cNx+OL
        atm_FWFlux(i,j)     = dP(fS%LoC(1)+i-1,fS%LoC(2)+j-1)
       ENDDO
      ENDDO
!
!     (3.) Execute MITgcm OCN internal runstep execution sequence.
!
      startStep = aCount
      stopStep  = aCount+1
      CALL OCN_RUN(
     I              atm_HeatFlux, atm_TauX, atm_TauY,
     I              atm_Qlatent, atm_Qsensible, atm_Qlongwave,
     I              atm_Qshortwave,
     I              atm_uVelGround, atm_vVelGround,
     I              atm_FWFlux,
     O              ocn_SSTocn,
     I              startStep, stopStep
     &             )

!     (4.) Update export state (filling in export state halo regions update)
!     mitgcm_org_ocn SSTocn
      CALL STATE_GET_FADP( eState, export_names( 2), dP , theFieldRef=fieldRef )
      DO J=1,cNy
       DO I=1, cNx
        dP(fS%LoC(1)+i-1,fS%LoC(2)+j-1) = ocn_SSTocn(i,j)
       ENDDO
      ENDDO
      CALL ESMF_FieldHalo( fieldRef )

!     Clean up temporaries
      DEALLOCATE( atm_HeatFlux   )
      DEALLOCATE( atm_TauX       )
      DEALLOCATE( atm_TauY       )
      DEALLOCATE( atm_Qlatent    )
      DEALLOCATE( atm_Qsensible  )
      DEALLOCATE( atm_Qlongwave  )
      DEALLOCATE( atm_Qshortwave )
      DEALLOCATE( atm_uVelground )
      DEALLOCATE( atm_vVelground )
      DEALLOCATE( atm_FWFlux     )
      DEALLOCATE( ocn_SSTocn     )
      RETURN
      END SUBROUTINE
      END MODULE
